/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   Framework Module Resource Definition
::   Copyright   :   (C)2002-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource.h
    \brief The framework encapsulates the module hardware which it then abstracts through \e behaviours and
           \e resources. Framework resources map to the module's I/O connections and thus each connection is
           described by a framework resource. The \ref E_ModuleResource enumeration describes the supported
           resources while \ref E_ResourceBehaviour describes the supported behaviours.
           
\section resourcesandhardware Resource and Hardware
The Framework is designed to support a family of MPC5xx based modules. These modules have different functionality
and capability and so a module will not support the entire set of resources described by \ref E_ModuleResource.
A single enumeration was used to describe those resources because it allows MPC5xx applications to be designed
that are completely decoupled from the modules that they will operate on. Thus there is the potential for same
application source code to be employed on two completely different modules. The framework will fail any attempt
to employ a resource that is not supported by the executing hardware and issue the
\c ERROR_MODULE_DOES_NOT_SUPPORT_HARDWARE native error code. The IsResourceSupportedByHardware() function can
be used to determine whether the resource is supported prior to making a call to the framework.

Resources are prefixed with \c RES_ followed by the name used on the module schematic. It should be possible to
to identify which framework resources are connected in a given application given a module specific block diagram
that describes the application's setup.

\section usingresources Using Resources and Behaviours
Resources provide the mechanism by which behaviours (which equate to functionality) can be mapped to a module's
I/O. Nearly all of the framework's API functions employ the \ref E_ModuleResource type. Behaviours are mapped
to resources via the \e Create functions and freed through \e Destroy functions. Note that many behaviours
can't be destroyed once created. Minimising the Framework's footprint is the main driving force behind why many
behaviours do not support the Destroy operation. The \c ERROR_UNSUPPORTED_IN_ARCHIVE native error code is
returned when the destroy functionality is not supported by a particular behaviour.

Most resource only support a subset of the resources available. Most input type resources will not support
output behaviours and vice versa. Some behaviours are also very specific. The \ref BEHAVIOUR_PROSAK_KNK behaviour
is but one example of this.

Most resources will only allow a single behaviour to be mapped at any one time. This is true for all output type
behaviours like \ref BEHAVIOUR_PWM. Assigning two behaviours to an output will result in conflict since it can't
know which output behaviour should be applied at any given time. The \c Destroy APIs need to be employed when
one output behaviour must be switched with another. The \c ERROR_RESOURCE_CONFLICT native error is returned
when an application attempts to map a second behaviour to a resource that does not allow such operation.

Some input behaviours, most notably \ref BEHAVIOUR_SWITCH and \ref BEHAVIOUR_ANALOGIN can be assigned to the
same resource. \ref BEHAVIOUR_ANALOGIN can also be assigned to a resource that is an output if hardware support 
exists.  Eg: RES_LSD1 may have current sensing, therefore can also be defined as an analogue input */

#ifndef __RESOURCE_H
#define __RESOURCE_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <NativeError.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/

/* Overloaded Resource Handling APIs */
#define CreateResource(RES,DATA_PTR,BEHAVIOUR)          CreateResource##BEHAVIOUR(RES,DATA_PTR)
#define SetResourceAttributes(RES,DATA_PTR,BEHAVIOUR)   SetResourceAttributes##BEHAVIOUR(RES,DATA_PTR)
#define GetResourceAttributes(RES,DATA_PTR,BEHAVIOUR)   GetResourceAttributes##BEHAVIOUR(RES,DATA_PTR)
#define DestroyResource(RES,BEHAVIOUR)                  DestroyResource##BEHAVIOUR(RES)
#define GetResourceValue(RES,HOW_TO_PTR,DATA_PTR,BEHAVIOUR)  GetResourceValue##BEHAVIOUR(RES,HOW_TO_PTR,DATA_PTR)

/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

#define RES_USB_PWR2 RES_XDRP_B
#define RES_HBRDG3 RES_ESC
/*! \brief The framework encapsulates the module hardware which it then abstracts through \e behaviours and
           \e resources. Framework resources map to the module's I/O connections and thus each connection is
           described by a framework resource. This enumeration describes all the potential resources.

Also see
- \ref resourcesandhardware
- \ref usingresources */

/* Name: E_ModuleResource ClassID:ENUMDEC EnumDflt:"Illegal" */
typedef enum
{
    RES_FUELP            = 0,     /* EnumTxt:"FUELP"          */
    RES_H1               = 1,     /* EnumTxt:"ETC (H1)"       */
    RES_HSOL1            = 2,     /* EnumTxt:"HSOL1"          */
    RES_HSOL2            = 3,     /* EnumTxt:"HSOL2"          */
    RES_HSOL3            = 4,     /* EnumTxt:"HSOL3"          */
    RES_HSOL4            = 5,     /* EnumTxt:"HSOL4"          */
    RES_START            = 6,     /* EnumTxt:"START"          */
    RES_LIAC             = 7,     /* EnumTxt:"LIAC"           */
    RES_HORN             = 8,     /* EnumTxt:"HORN"           */
    RES_SPARE            = 9,     /* EnumTxt:"SPARE"          */
    RES_H2               = 10,    /* EnumTxt:"ETC1 (H2)"      */
    RES_OILP             = 11,    /* EnumTxt:"OILP"           */
    RES_INJ1D            = 12,    /* EnumTxt:"INJ1"           */
    RES_INJ2D            = 13,    /* EnumTxt:"INJ2"           */
    RES_INJ3D            = 14,    /* EnumTxt:"INJ3"           */
    RES_INJ4D            = 15,    /* EnumTxt:"INJ4"           */
    RES_INJ5D            = 16,    /* EnumTxt:"INJ5"           */
    RES_INJ6D            = 17,    /* EnumTxt:"INJ6"           */
    RES_INJ7D            = 18,    /* EnumTxt:"INJ7"           */
    RES_INJ8D            = 19,    /* EnumTxt:"INJ8"           */
    RES_INJ9D            = 20,    /* EnumTxt:"INJ9"           */
    RES_INJ10D           = 21,    /* EnumTxt:"INJ10"          */
    RES_INJ11D           = 22,    /* EnumTxt:"INJ11"          */
    RES_INJ12D           = 23,    /* EnumTxt:"INJ12"          */
    RES_TACH             = 24,    /* EnumTxt:"TACH"           */
    RES_LSD1             = 25,    /* EnumTxt:"LSD1"           */
    RES_LSD2             = 26,    /* EnumTxt:"LSD2"           */
    RES_LSD3             = 27,    /* EnumTxt:"LSD3"           */
    RES_LSD4             = 28,    /* EnumTxt:"LSD4"           */
    RES_LSD5             = 29,    /* EnumTxt:"LSD5"           */
    RES_LSD6             = 30,    /* EnumTxt:"LSD6"           */
    RES_LSD7             = 31,    /* EnumTxt:"LSD7"           */
    RES_LSD8             = 32,    /* EnumTxt:"LSD8"           */
    RES_LSD9             = 33,    /* EnumTxt:"LSD9"           */
    RES_LSD10            = 34,    /* EnumTxt:"LSD10"          */
    RES_ESC              = 35,    /* EnumTxt:"ESC"            */
    RES_EST1             = 36,    /* EnumTxt:"EST1"           */
    RES_EST2             = 37,    /* EnumTxt:"EST2"           */
    RES_EST3             = 38,    /* EnumTxt:"EST3"           */
    RES_EST4             = 39,    /* EnumTxt:"EST4"           */
    RES_EST5             = 40,    /* EnumTxt:"EST5"           */
    RES_EST6             = 41,    /* EnumTxt:"EST6"           */
    RES_EST7             = 42,    /* EnumTxt:"EST7"           */
    RES_EST8             = 43,    /* EnumTxt:"EST8"           */
    RES_EST9             = 44,    /* EnumTxt:"EST9"           */
    RES_EST10            = 45,    /* EnumTxt:"EST10"          */
    RES_EST11            = 46,    /* EnumTxt:"EST11"          */
    RES_EST12            = 47,    /* EnumTxt:"EST12"          */
    RES_EST13            = 48,    /* EnumTxt:"EST13"          */
    RES_EST14            = 49,    /* EnumTxt:"EST14"          */
    RES_EST15            = 50,    /* EnumTxt:"EST15"          */
    RES_EST16            = 51,    /* EnumTxt:"EST16"          */
    RES_MPRD             = 52,    /* EnumTxt:"MPRD"           */
    RES_ECUP             = 53,    /* EnumTxt:"ECUP"           */
    RES_DRVP             = 54,    /* EnumTxt:"DRVP"           */
    RES_XDRP             = 55,    /* EnumTxt:"XDRP"           */
    RES_AN1M             = 56,    /* EnumTxt:"AN1M"           */
    RES_AN2M             = 57,    /* EnumTxt:"AN2M"           */
    RES_AN3M             = 58,    /* EnumTxt:"AN3M"           */
    RES_AN4M             = 59,    /* EnumTxt:"AN4M"           */
    RES_AN5M             = 60,    /* EnumTxt:"AN5M"           */
    RES_AN6M             = 61,    /* EnumTxt:"AN6M"           */
    RES_AN7M             = 62,    /* EnumTxt:"AN7M"           */
    RES_AN8M             = 63,    /* EnumTxt:"AN8M"           */
    RES_AN9M             = 64,    /* EnumTxt:"AN9M"           */
    RES_AN10M            = 65,    /* EnumTxt:"AN10M"          */
    RES_AN11M            = 66,    /* EnumTxt:"AN11M"          */
    RES_AN12M            = 67,    /* EnumTxt:"AN12M"          */
    RES_AN13M            = 68,    /* EnumTxt:"AN13M"          */
    RES_AN14M            = 69,    /* EnumTxt:"AN14M"          */
    RES_AN15M            = 70,    /* EnumTxt:"AN15M"          */
    RES_AN16M            = 71,    /* EnumTxt:"AN16M"          */
    RES_AN17M            = 72,    /* EnumTxt:"AN17M"          */
    RES_AN18M            = 73,    /* EnumTxt:"AN18M"          */
    RES_AN19M            = 74,    /* EnumTxt:"AN19M"          */
    RES_AN20M            = 75,    /* EnumTxt:"AN20M"          */
    RES_AN21M            = 76,    /* EnumTxt:"AN21M"          */
    RES_AN22M            = 77,    /* EnumTxt:"AN22M"          */
    RES_AN23M            = 78,    /* EnumTxt:"AN23M"          */
    RES_AN24M            = 79,    /* EnumTxt:"AN24M"          */
    RES_AN25M            = 80,    /* EnumTxt:"AN25M"          */
    RES_AN26M            = 81,    /* EnumTxt:"AN26M"          */
    RES_AN27M            = 82,    /* EnumTxt:"AN27M"          */
    RES_AN28M            = 83,    /* EnumTxt:"AN28M"          */
    RES_AN29M            = 84,    /* EnumTxt:"AN29M"          */
    RES_AN30M            = 85,    /* EnumTxt:"AN30M"          */
    RES_AN31M            = 86,    /* EnumTxt:"AN31M"          */
    RES_AN32M            = 87,    /* EnumTxt:"AN32M"          */
    RES_AN33M            = 88,    /* EnumTxt:"AN33M"          */
    RES_AN34M            = 89,    /* EnumTxt:"AN34M"          */
    RES_O2A              = 90,    /* EnumTxt:"O2A"            */
    RES_O2B              = 91,    /* EnumTxt:"O2B"            */
    RES_XDRP_B           = 92,    /* EnumTxt:"XDRP_B"         */
    RES_DG1M             = 93,    /* EnumTxt:"DG1M"           */
    RES_DG2M             = 94,    /* EnumTxt:"DG2M"           */
    RES_DG3M             = 95,    /* EnumTxt:"DG3M"           */
    RES_DG4M             = 96,    /* EnumTxt:"DG4M"           */
    RES_STOP             = 97,    /* EnumTxt:"STOP"           */
    RES_PROSAK           = 98,    /* EnumTxt:"Knock"          */
    RES_PROSAK2          = 99,    /* EnumTxt:"Knock 2"        */
    RES_ENCODER          = 100,   /* EnumTxt:"Encoder"        */
    RES_CAN1             = 101,   /* EnumTxt:"CAN1"           */
    RES_DG5M             = 102,   /* EnumTxt:"DG5M"           */
    RES_DG6M             = 103,   /* EnumTxt:"DG6M"           */
    RES_DG7M             = 104,   /* EnumTxt:"DG7M"           */
    RES_SPD1             = 105,   /* EnumTxt:"SPD1"           */
    RES_SPD2             = 106,   /* EnumTxt:"SPD2"           */
    RES_H3               = 107,   /* EnumTxt:"H3"             */
    RES_INJCURR_SELECT   = 108,   /* EnumTxt:"Inj Current Sel"*/
    RES_CJ125_LSU1       = 109,   /* EnumTxt:"CJ125 LSU 1"    */
    RES_CJ125_LSU2       = 110,   /* EnumTxt:"CJ125 LSU 2"    */
    RES_DG8M             = 111,   /* EnumTxt:"DG8M"           */
    RES_ISO9141_L        = 112,   /* EnumTxt:"ISO9141_L"      */
    RES_RS485            = 113,   /* EnumTxt:"RS485 Link"     */
    RES_ISO9141_K        = 114,   /* EnumTxt:"ISO9141K Link"  */
    RES_CAN2             = 115,   /* EnumTxt:"CAN2"           */
    RES_RS232            = 116,   /* EnumTxt:"RS232 Link"     */
    RES_MODULE_TEMP      = 117,   /* EnumTxt:"Module Temp"    */
    RES_LED1             = 118,   /* EnumTxt:"LED1"           */
    RES_LED2             = 119,   /* EnumTxt:"LED2"           */
    RES_BOOT             = 120,   /* EnumTxt:"BOOT"           */
    RES_BKUP_BATT        = 121,   /* EnumTxt:"BKUP_BATT"      */
    RES_CAM              = 122,   /* EnumTxt:"CAM"            */
    RES_TISS             = 123,   /* EnumTxt:"TISS"           */
    RES_TOSS             = 124,   /* EnumTxt:"TOSS"           */
	RES_LSD11			 = 125,   /* EnumTxt:"LSD11"          */
	RES_LSD12			 = 126,   /* EnumTxt:"LSD12"          */
	RES_LSD13			 = 127,   /* EnumTxt:"LSD13"          */
	RES_LSD14			 = 128,   /* EnumTxt:"LSD14"          */
	RES_LSD15			 = 129,   /* EnumTxt:"LSD15"          */
	RES_LSD16			 = 130,   /* EnumTxt:"LSD16"          */
	RES_LSD17			 = 131,   /* EnumTxt:"LSD17"          */
	RES_LSD18			 = 132,   /* EnumTxt:"LSD18"          */
    RES_CAN3             = 133,   /* EnumTxt:"CAN3"           */
    RES_H1_PLUS          = 134,   /* EnumTxt:"LSO_H1+"        */
    RES_H1_MINUS         = 135,   /* EnumTxt:"LSO_H1-"        */
    RES_H2_PLUS          = 136,   /* EnumTxt:"LSO_H2+"        */
    RES_H2_MINUS         = 137,   /* EnumTxt:"LSO_H2-"        */
	RES_ETRIG1		     = 138,   /* EnumTxt:"ETRIG1 Output resource" */
	RES_ETRIG2		     = 139,   /* EnumTxt:"ETRIG2 Output resource" */
	RES_HSD1		     = 140,   /* EnumTxt:"HSD1" */
	RES_HSD2		     = 141,   /* EnumTxt:"HSD2" */
    RES_PWMADC           = 142,   /* EnumTxt:"PWMADC Used to check that the CCM is providing valid data" */

/* Resources that don't get repeated in Resource Layout Files */
    RES_14V_REF,


    /* If you add a resource here YOU MUST ADD that same resource entry in the Resource_Defn.c file */
    /* Monotonically increasing indexing is assumed                                                 */
    NUM_SUPPORTED_RESOURCES,

    /* These resources are handled differently and defining after NUM_SUPPORTED_RESOURCES conserves memory */
    RES_SCHEDULER,
    RES_SCHEDULER_2,
    RES_SCHEDULER_3,
    RES_ANGLE_EVENT_1,
    RES_TCPIPSTACK,
    RES_USB_P1_H2,                /* Peripheral on Port 1 and Host on Port 2 */
	RES_ADCTIME,				  /* EnumTxt:"Default timebased samplig QADC queue resource" */
	RES_ADCTRIG_A,                 /* EnumTxt:"Ext Triggered ANALOG_TRIGGER_QUEUE1_A queue resource" */
	RES_ADCTRIG_B,                 /* EnumTxt:"Ext Triggered ANALOG_TRIGGER_QUEUE1_B queue resource" */
} E_ModuleResource;

/*! \brief The framework encapsulates the module hardware which it then abstracts through \e behaviours and
           \e resources. Each of the major functionalities offered by the framework are described by the
           behaviours of this enumeration. A behaviour is instantiated by assigning it to a resource via
           the approriate create function.

Also see
- \ref resourcesandhardware
- \ref usingresources */

/* Name: E_ResourceBehaviour ClassID:ENUMDEC EnumDflt:"Illegal"*/
typedef enum
{
    BEHAVIOUR_NONE,

/*! Pulse Width Modulation behaviour that can be applied to an output resource. See \ref Resource_PWM.c */
    BEHAVIOUR_PWM,            /* EnumTxt:"PWM"                 */
/*! A One Shot output operates like a monostable. It can be setup to turn ON for a defined amount of time
    before automatically turning OFF again. See \ref Resource_OneShot.c */
    BEHAVIOUR_ONESHOT,        /* EnumTxt:"OneShot"             */
/*! Discrete output behaviour most notably used to control relays. This is simplest output behaviour.
    See \ref Resource_DiscreteOut.c */
    BEHAVIOUR_DISCRETE_OUT,   /* EnumTxt:"Discrete Output"     */
/*! Frequency input behaviour. See \ref Resource_FreqIn.c */
    BEHAVIOUR_FREQIN,         /* EnumTxt:"Frequency Input"     */
/*! Switch input behaviour. See \ref Resource_Switch.c */
    BEHAVIOUR_SWITCH,         /* EnumTxt:"Switch"              */
/*! Sequence output behaviour that has been specially tailored to meet the needs of transient fuel injection.
    See \ref transientfuelseqout */
    BEHAVIOUR_TRAN_FUEL_SEQ,  /* EnumTxt:"Transient Injector"  */
/*! Sequence output behaviour that has been specially tailored to meet the needs of banked ignition.
    See \ref bankedestseqout */
    BEHAVIOUR_BANK_EST_SEQ,   /* EnumTxt:"Banked Spark"        */
/*! Sequence output behaviour that has the capability to superimpose multiple pulses on one output resource.
    See \ref multiplexedseqout */
    BEHAVIOUR_MUX_EST_SEQ,    /* EnumTxt:"Multiplexed Spark"   */
/*! Sequence output behaviour that can be used to control injectors. See \ref injectorseqout */
    BEHAVIOUR_INJ_SEQ,        /* EnumTxt:"Injector"            */
/*! Sequence output behaviour that can be used to control an ignition coil. See \ref estseqout */
    BEHAVIOUR_EST_SEQ,        /* EnumTxt:"Spark"               */
/*! Behaviour that allows an analogue input to be read. See Resource_Analog.c */
    BEHAVIOUR_ANALOGIN,       /* EnumTxt:"Analog Input"        */
/*! Behaviour that allows an analogue input to be sampled on a Trigger signal. See Resource_Trigger.c */
    BEHAVIOUR_ANALOG_TRIGGER,    /* EnumTxt:"Triggered Analog Input support"        */
/*! Behaviour that allows the single wire EZlink communications link to be created on an output. */
    BEHAVIOUR_EZLINK,         /* EnumTxt:"EZLink Gauge"        */
/*! ProSAK knock ASIC control. See \ref Resource_knock.c */
    BEHAVIOUR_PROSAK_KNK,
/*! Multiple pulse tachometer output behaviour that can be used on some output resource. See \ref Resource_Tacho.c */
    BEHAVIOUR_TACHO,          /* EnumTxt:"Tacho"               */
/*! Wide Range Lambda sensing unit behaviour for the CJ125 ASIC. See \ref Resource_CJ125_LSU.c */
    BEHAVIOUR_CJ125_LSU,      /* EnumTxt:"CJ125 LSU" */
/*! Task scheduler service behaviour. Applications using the Framework are expected to be event driven. The
    framework scheduler manages the issuing of the events and the passing of execution control to the
    application so that it may act upon them. This behaviour allows the creation of that scheduler. Multiple
    schedulers can be created. All framework and user defined events are fed through each instantiated
    scheduler so as to allow for application task execution. Also see \ref Resource_Scheduler.c */
    BEHAVIOUR_SCHEDULER,
/*! The Framework is responsible for analysing the crank angle position information made available to it by the
    crank position sensor (CPS) and/or cam sensor. The combination of crank and cam sensors is referred to as
    the encoder. The method allows the application to configure the Framework's detection of crank position,
    as well as to allow task scheduling and provide the application software with information like engine speed.
    See \ref Resource_Encoder.c */
    BEHAVIOUR_ENCODER,
/*! The BEHAVIOUR_TOUCAN behaviour abstracts the MPC5xx TouCAN hardware. It allows CAN messages to be sent and
    received using this hardware. */
    BEHAVIOUR_TOUCAN,
/*! Behaviour allows the control of injector current peak and hold limits. */
    BEHAVIOUR_INJCURR_SELECT,
/*! The behaviour exposes the framework's communication interface behaviour. The interface is modelled on a
    universal asynchronous receiver transmitter (UART) type serial communications interface resource. An
    application can, through this behaviour, reserve a serial communications resource and then use it to
    send and receive data over the communications link. */
    BEHAVIOUR_COMM,
/*! The behaviour exposes the framework's angle event generation interface. The application can use this behaviour
    to obtain notification when a particular angle is observed */
    BEHAVIOUR_ANGLE_EVENT,
/*! The behaviour exposes the framework's LED control interface. */
    BEHAVIOUR_LED,
/*! The behaviour exposes the framework's TCPIP stack. */
    BEHAVIOUR_TCPIP,
/*! The behaviour exposes the framework's USB stack. */
    BEHAVIOUR_USB,
/*! Variable cam phasor behaviour */
    BEHAVIOUR_VARCAM,
/*! Transition Counter behaviour */
    BEHAVIOUR_TRANSCOUNT,
/*! Encoder Emulator */
    BEHAVIOUR_ENC_EMULATOR,
    NUM_BEHAVIOURS,
} E_ResourceBehaviour;

/*! \brief The condition attribute defines the operational status of the resource.

A \ref RES_DISABLED indicates that the resource is not operational while \ref RES_ENABLED indicates that a
resource is operational. The output stage of an output resource is placed in the inactive (OFF) state when
it is disabled. The attributes of a disabled resource can still be affected through the appropriate function
calls, but those attributes will not be "applied" to the resource until it becomes operational again.
Operational events, like the occurance of an edge for example, are ignored while a resource is disabled. */
/* Name: E_ResourceCond ClassID:ENUMDEC EnumDflt:"Illegal"     */
typedef enum
{
/*! Inoperational state. */
    RES_DISABLED = 0,                  /*  EnumTxt:"Disabled"  */
/*! Operational state. */
    RES_ENABLED = 1                    /*  EnumTxt:"Enabled"   */
} E_ResourceCond;

/*! \brief Universal polarity attribute. Polarity is always with respect to the pin connection. Any inversion
           that may result within the module because of hardware signal pre-processing is implicitly handled
           by the framework. */
/* Name: E_ResourcePolarity ClassID:ENUMDEC EnumDflt:"Illegal" */
typedef enum
{
/*! Resource has an active low polarity. The active condition is applied while the pin is low. A switch with
    this polarity would have a condition of RES_OFF when the pin was at it's logic-1 level (\ref RES_PINSTATE_HIGH) */
    RES_POL_ACTIVE_LOW,                /* EnumTxt:"Active Low" */
/*! Resource has an active high polarity. The active condition is applied while the pin is high. A switch with
    this polarity would have a condition of RES_OFF when the pin was at it's logic-0 level (\ref RES_PINSTATE_LOW) */
    RES_POL_ACTIVE_HIGH                /* EnumTxt:"Active High"*/
} E_ResourcePolarity;

/*! \brief Universal pin state attribute. Pin state is always with respect to the pin connection. Any inversion
           that may result within the module because of hardware signal pre or post processing is implicitly
           handled by the framework. */
/* Name: E_ResourcePinState ClassID:ENUMDEC EnumDflt:"Illegal" */
typedef enum
{
/*! The input pin is low or at its logic-0 state. */
    RES_PINSTATE_LOW,                  /* EnumTxt:"LOW"        */
/*! The input pin is high or at its logic-1 state. */
    RES_PINSTATE_HIGH                  /* EnumTxt:"HIGH"       */
} E_ResourcePinState;

/*! \brief State attribute that fundamentally indicates whether the resource is ON or OFF. */
/* Name: E_ResourceState ClassID:ENUMDEC EnumDflt:"Illegal"    */
typedef enum
{
/*! The resource is OFF */
    RES_OFF,                           /* EnumTxt:"OFF"        */
/*! The resource is ON, and when the resource is directional, is also forward */
    RES_ON,                            /* EnumTxt:"ON"         */
/*! The direction resource is ON, and in reverse. */
    RES_ON_REVERSE,
} E_ResourceState;

/*! Describes the diagnostic status of a resource.

    Many hardware diagnostic schemes are only able to identify short circuits while on and open circuits while
    they are off. For example, an indeterminate result may be supplied if an output was currently turned ON,
    but its output was open circuit. Thus while the output is not shorted, it is not known whether the output
    is open. This is why the indeterminate value could be supplied. */

/* Name: E_ResourceDiagStatus ClassID:ENUMDEC EnumDflt:"Illegal"   */
typedef enum
{
/*! It is not possible to determine whether the hardware described by the resource is currently in a fault
    free state. */
    RES_STATUS_INDETERMINATE,           /* EnumTxt:"Indeterminate" */
/*! The hardware described by this resource is currently in a fault condition. */
    RES_STATUS_FAULT,                   /* EnumTxt:"Fault"         */
/*! The hardware described by this resource is currently in a fault free condition. */
    RES_STATUS_OK                       /* EnumTxt:"OK"            */
} E_ResourceDiagStatus;

/*! Describes the edge sensitivity of a digital input resource.

    Sensitivity is always taken with respect to the module's input */

/* Name: E_ResourceEdgeSensitivity ClassID:ENUMDEC EnumDflt:"Illegal"   */
typedef enum
{
/*! Rising edge at the module's input is detected. */
    RES_RISE_EDGE_DETECT,   /* EnumTxt:"Detect Rising Edge" */
/*! Falling edge at the module's input is detected. */
    RES_FALL_EDGE_DETECT,   /* EnumTxt:"Detect Falling Edge" */
/*! Rising and falling edges are detected as the module's input */
    RES_EITHER_EDGE_DETECT, /* EnumTxt:"Detect Either Edge" */
} E_ResourceEdgeSensitivity;

/*! \brief Some of Woodward's MPC5xx controllers */
typedef enum
{
/*! Based on the same microcontroller technology as the PCM, the Woodward Engine Control Module provides most
    of the same basic capabilities but in a smaller, more economical package. The ECM is aimed primarily at
    engine-only control applications and uses a smaller, 48-pin waterproof connector. Because it is based on
    the same microcontroller, it is capable of running the same software set used by the PCM, albeit with
    slightly less I/O capacity. A single MotoNet port is available for linking to a variety of Woodward's
    electronic display and gauge products. */ 
    MODULE_ECM,
/*! Full-featured, powerful and rugged, yet light and compact, the Woodward Powertrain Control Module has the
    capability to address the most difficult powertrain control problems. PCM makes use of the Motorola MPC555
    microcontroller and ProSAK knock control chip - each the most capable of its type. Engine mountable, with
    an 80-pin waterproof connector, it provides plenty of I/O capability for most applications. Also has is the
    extensible networking capability of MotoNet, a dual CAN bus architecture for both information sharing and
    multiplexed real-time control. */
    MODULE_PCM,
/*! Based on the same technology as the PCM, Woodward's Racing version of the Powertrain Control Module has
    all the capability of that controller, plus further I/O and emission control features. Includes the
    accessibility to dual ProSAK knock control, dual wide range lambda sensing, O2 control and the processing
    power of the Motorola MPC565 microcontroller. Has 128 pins of I/O housed in a waterproof connector. Also has
    is the extensible networking capability of MotoNet, a dual CAN bus architecture for both information sharing
    and multiplexed real-time control. Also wih on-board RAM extension and parallel EEPROM. */
    MODULE_PCMRACE_LOVOL,
/*! A different version of the PCM Race controller that sacrifices some O2 and knock capability to allow for more
    general I/O functionality. */
    MODULE_PCMRACE_HIVOL,
/*! InService logging module. */
    MODULE_ISDL,
/*! AIM / TVM (Thrust Vector Module) controller */
	MODULE_AIM,
/*! ECM48 Model 0701 utilizing the MPC563 microcontroller  for Industrial Applications */ 
    MODULE_ECM48_0701,
/*! ECM Module Replacement utilizing the MPC563 microcontroller for Marine Applications */ 
    MODULE_ECM48_0702,
/*! ECM Module Replacement utilizing the MPC563 microcontroller for Industrial Applications */ 
    MODULE_ECM48_0703,
/*! FIRE48 CCM / SIM */
    MODULE_FIRE48_CCM,
    MODULE_FIRE48_SIM,
/*! TVM48  TVM / HLA / MII */
    MODULE_TVM48_TVM,
    MODULE_TVM48_HLA,
    MODULE_TVM48_MII,
} E_ModuleHardwareType;

/*! \brief Defines the operating mode of a driver. Some modules support drivers that can operate in more than one mode */
typedef enum
{
    DRIVER_LOW_SIDE,    /*!< Low side driver configuration that is typically used with a high side load to battery.
                             The \ref RES_OFF state puts the low side switch into a high impedance state while the
                             \ref RES_ON state puts the switch into a low impedance state that will sink current. */
    DRIVER_PUSH_PULL,   /*!< Low current logic level push pull style driver. The \ref RES_OFF state has the low side
                             actively pulling to ground where it will act as a current sink. Connection of a high side
                             load to such a driver is not expected as the drive will conduct current through that load
                             while it is OFF. */
	DRIVER_HIGH_SIDE,	/*!< High side driver configuration that is typically used with a load to GND.
						     The \ref RES_OFF state puts the high side switch into a high impedance state while the
						     \ref RES_ON state puts the switch into a low impedance state that will sink current. */

} E_DriverMode;

/*! \brief Detection threshold that is applied to some inputs, most notably VR interfaces */

/* Name: E_DetectionThreshMode ClassID:ENUMDEC EnumDflt:"Illegal"   */
typedef enum
{
/*! Detection threshold is fixed by hardware */
    THRESH_FIXED_HW,        /* EnumTxt:"Fixed HW Threshold"         */
/*! Detection threshold is adaptive through hardware */
    THRESH_ADAPT_HW         /* EnumTxt:"Adaptive HW Threshold"      */
} E_DetectionThreshMode;

/*! Defines the possible software configurable pull-up settings for the encoder */
/* Name: E_EncoderPullUp ClassID:ENUMDEC EnumDflt:"Illegal Pullup" */
typedef enum
{
/*! Weak pullup setting */
    PULLUP_R_WEAK = 0,              /* EnumTxt:"Weak Pullup" */
/*! Strong pullup setting */
    PULLUP_R_STRONG = 1,            /* EnumTxt:"Strong Pullup" */
} E_EncoderPullUp;

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid attributes
           in the S_ResourceAllocateFW data structure which is used with AllocateResourcesInFW().

Each mask definition applies to a particular member of one of the data structures.
\code
    ...
    S_ResourceAllocateFW AttribsObj;

    // Going to provide values for the oneshots and pwms and will use the framework default for the others
    AttribsObj.uValidAttributesMask = USE_ALLOCATE_ONESHOTS | USE_ALLOCATE_PWMS;
    AttribsObj.uNumOneshotsToAllocate = 0;  // Application won't be using the oneshot behaviour
    AttribsObj.uNumPWMsToAllocate = 10;     // Application will need upto 10 PWMs
    // The remainder of the data structure can be left undefined since the uValidAttributesMask
    //   indicates that the members are not valid
    ... \endcode */
typedef enum
{
    USE_ALLOCATE_ONESHOTS = 0x01,       /*!< Selects S_ResourceAllocateFW::uNumOneshotsToAllocate */
    USE_ALLOCATE_PWMS = 0x02,           /*!< Selects S_ResourceAllocateFW::uNumPWMsToAllocate */
    USE_ALLOCATE_DISCRETEOUTS = 0x04,   /*!< Selects S_ResourceAllocateFW::uNumDiscreteOutsToAllocate */
    USE_ALLOCATE_VARCAMS = 0x08,        /*!< Selects S_ResourceAllocateFW::uNumVarCamsToAllocate */
    USE_ALLOCATE_SWITCHES = 0x10,       /*!< Selects S_ResourceAllocateFW::uNumSwitchesToAllocate */
    USE_ALLOCATE_FREQINS = 0x20,        /*!< Selects S_ResourceAllocateFW::uNumFreqInsToAllocate */
    USE_ALLOCATE_LEDS = 0x40,           /*!< Selects S_ResourceAllocateFW::uNumLEDsToAllocate */
    USE_ALLOCATE_TRANSCOUNT = 0x80,     /*!< Selects S_ResourceAllocateFW::uNumTransCountToAllocate */
    /* Alter the type of AllocateType_U if the number of attributes exceeds 8 */
} E_ResourceAllocateMask;

typedef uint1 AllocateType_U;

typedef struct
{
/*! Logic-OR the attributes [\ref USE_ALLOCATE_ONESHOTS, \ref USE_ALLOCATE_PWMS, \ref USE_ALLOCATE_DISCRETEOUTS, \ref USE_ALLOCATE_SWITCHES,
    and \ref USE_ALLOCATE_FREQINS ]
    that are valid for this instance of the data structure */
    AllocateType_U uValidAttributesMask;
/*! Defines the amount of memory that the framework archive should reserve to support BEHAVIOUR_ONESHOT and
    its associated resources. Select with the \ref USE_ALLOCATE_ONESHOTS bit mask. */
    uint1 uNumOneshotsToAllocate;
/*! Defines the amount of memory that the framework archive should reserve to support \ref BEHAVIOUR_PWM and
    its associated resources. Select with the \ref USE_ALLOCATE_PWMS bit mask. */
    uint1 uNumPWMsToAllocate;
/*! Defines the amount of memory that the framework archive should reserve to support the BEHAVIOUR_DISCRETE_OUTS
    and its associated resources. Select with the \ref USE_ALLOCATE_DISCRETEOUTS bit mask. */
    uint1 uNumDiscreteOutsToAllocate;
/*! Defines the amount of memory that the framework archive should reserve to support the BEHAVIOUR_SWITCH and
    its associated resources. Select with the \ref USE_ALLOCATE_SWITCHES bit mask. */
    uint1 uNumSwitchesToAllocate;
/*! Defines the amount of memory that the framework archive should reserve to support the BEHAVIOUR_FREQIN and
    its associated resources. Select with the \ref USE_ALLOCATE_FREQINS bit mask. */
    uint1 uNumFreqInsToAllocate;
/*! Defines the amount of memory that the framework archive should reserve to support the BEHAVIOUR_LED and
    its associated resources. Select with the \ref USE_ALLOCATE_LEDS bit mask. */
    uint1 uNumLEDsToAllocate;

/*! Defines the amount of memory that the framework archive should reserve to support the BEHAVIOUR_VARCAM
    and its associated resources. Select with the \ref USE_ALLOCATE_VARCAMS bit mask. */
    uint1 uNumVarCamsToAllocate;

/*! Defines the amount of memory that the framework archive should reserve to support the BEHAVIOUR_TRANSCOUNT
    and its associated resources. Select with the \ref USE_ALLOCATE_TRANSCOUNT bit mask. */
    uint1 uNumTransCountToAllocate;

} S_ResourceAllocateFW;

#pragma pack()

/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/* Want to expose all the resource level APIs Here to ensure that the overloading of the resource handling */
/*   functions works as desired at the application level                                                   */
/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
#include <Resource_Trigger.h>
#include <Resource_Analog.h>
#include <Resource_DiscreteOut.h>
#include <Resource_PWM.h>
#include <Resource_OneShot.h>
#include <Resource_Tacho.h>
#include <Resource_EZLink.h>
#include <Resource_Switch.h>
#include <Resource_FreqIn.h>
#include <Resource_Knock.h>
#include <Resource_InjCurrSelect.h>
#include <Resource_EncoderEmulator.h>
#include <Resource_Encoder.h>
#include <Resource_SequenceOut.h>
#include <Resource_CJ125_LSU.h>
#include <Resource_Scheduler.h>
#include <Resource_Comm.h>
#include <Resource_CAN.h>
#include <Resource_AngleEvent.h>
#include <Resource_LED.h>
#include <Resource_FFS.h>
#include <Resource_TCPIP.h>
#include <Resource_TransCount.h>
#include <Resource_USB.h>
#include <Resource_VarCam.h>
#include <RealTimeClock.h>

/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- FUNCTION PROTOTYPES -------------------------------------------------------------------------------*/
NativeBool IsResourceSupportedByHardware(E_ModuleResource in_eResource);
E_ModuleHardwareType GetModuleHardwareType(void);
NativeError_S AllocateResourcesInFW(S_ResourceAllocateFW const* in_pAllocateConfig);

#endif /* __RESOURCE_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
